<?php

namespace Drupal\cms_content_sync\Plugin;

use Drupal\cms_content_sync\Entity\Flow;
use Drupal\cms_content_sync\PullIntent;
use Drupal\cms_content_sync\PushIntent;
use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use EdgeBox\SyncCore\Interfaces\Configuration\IDefineEntityType;
use EdgeBox\SyncCore\V2\Configuration\DefineProperty;
use EdgeBox\SyncCore\V2\Raw\Model\RemoteEntityListResponse;
use EdgeBox\SyncCore\V2\Raw\Model\RemoteEntitySummary;
use EdgeBox\SyncCore\V2\Raw\Model\RemoteRequestQueryParamsEntityList;

/**
 * The entity handler interface.
 *
 * Specifies the publicly available methods of an entity handler plugin that can
 * be used to push and pull entities with Sync Core.
 *
 * @see \Drupal\cms_content_sync\Annotation\EntityHandler
 * @see \Drupal\cms_content_sync\Plugin\EntityHandlerBase
 * @see \Drupal\cms_content_sync\Plugin\Type\EntityHandlerPluginManager
 * @see \Drupal\cms_content_sync\Entity\Flow
 * @see plugin_api
 *
 * @ingroup third_party
 */
interface EntityHandlerInterface extends PluginInspectionInterface {

  /**
   * Check if this handler supports the given entity type.
   *
   * @param string $entity_type
   *   The entity type to check.
   * @param string $bundle
   *   The entity bundle to check.
   *
   * @return bool
   *   True if the entity type is supported.
   */
  public static function supports($entity_type, $bundle);

  /**
   * Get the allowed push options.
   *
   * Get a list of all allowed push options for this entity.
   *
   * @see Flow::PUSH_*
   *
   * @return string[]
   *   Returns an array of allowed push options.
   */
  public function getAllowedPushOptions();

  /**
   * Get the allowed pull options.
   *
   * Get a list of all allowed pull options for this field.
   *
   * @see Flow::PULL_*
   *
   * @return string[]
   *   Returns an array of allowed pull options.
   */
  public function getAllowedPullOptions();

  /**
   * Get the allowed preview options.
   *
   * @return string[]
   *   Provide the allowed preview options used for display when manually
   *                  pulling entities
   */
  public function getAllowedPreviewOptions();

  /**
   * Get the handler settings.
   *
   * Return the actual form elements for any additional settings for this
   * handler.
   *
   * @param array $current_values
   *   The current values that the user set, if any.
   * @param string $type
   *   One of 'pull', 'push', 'both'.
   *
   * @return array
   *   Returns the settings for this handler.
   */
  public function getHandlerSettings(array $current_values, $type = 'both');

  /**
   * Validate the settings defined above.
   *
   * $form and $form_state are the same as in the Form API.
   *
   * @param array $form
   *   The form array to validate.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state to validate.
   * @param string $entity_type_name
   *   The entity type name to validate the handler settings for.
   * @param string $bundle_name
   *   The entity type bundle name to validate the handler settings for.
   * @param array $current_values
   *   The current values to validate.
   *
   * @return mixed
   *   Returns true on success.
   */
  public function validateHandlerSettings(array &$form, FormStateInterface $form_state, string $entity_type_name, string $bundle_name, array $current_values);

  /**
   * Update the entity type definition.
   *
   * Advanced entity type definition settings for the Sync Core. You
   * can usually ignore these.
   *
   * @param \EdgeBox\SyncCore\Interfaces\Configuration\IDefineEntityType|EdgeBox\SyncCore\V2\Configuration\DefineProperty $definition
   *   The definition to be sent to Sync Core.
   *                                                                                 {@see SyncCoreExport}.
   */
  public function updateEntityTypeDefinition(IDefineEntityType|DefineProperty &$definition);

  /**
   * Provide a list of fields that are not allowed to be pushed or pulled.
   *
   * These fields typically contain all label fields that are pushed
   * separately anyway (we don't want to set IDs and revision IDs of entities
   * for example, but only use the UUID for references).
   *
   * @return string[]
   *   Returns a string or an array of forbidden fields.
   */
  public function getForbiddenFields();

  /**
   * Pull an entity from the sync core.
   *
   * @throws \Drupal\cms_content_sync\Exception\SyncException
   *
   * @return bool
   *   Whether or not the content has been pulled. FALSE is a desired state,
   *              meaning nothing should be pulled according to config.
   */
  public function pull(PullIntent $intent);

  /**
   * Push an entity to the sync core.
   *
   * @param \Drupal\cms_content_sync\PushIntent $intent
   *   The request to store all relevant info at.
   *
   * @throws \Drupal\cms_content_sync\Exception\SyncException
   *
   * @return bool
   *   Whether or not the content has been pushed. FALSE is a desired state,
   *              meaning nothing should be pushed according to config.
   */
  public function push(PushIntent $intent);

  /**
   * Whether the entity of this intent is allowed to be pushed.
   *
   * @return bool
   *   Returns false of the entity should be ignored.
   */
  public function ignorePush(PushIntent $intent);

  /**
   * Get the entity view url.
   *
   * @return string
   *   Returns the entity view url.
   */
  public function getViewUrl(EntityInterface $entity);

  /**
   * Return the requested page of entities based on the given query object.
   * This is only used if the entity type + bundle are known. If not, the status
   * entities will be used to query the entities and the ->getSyncCoreListItem()
   * will be called per item.
   *
   * @param \Drupal\cms_content_sync\Entity\Flow $flow
   * @param \EdgeBox\SyncCore\V2\Raw\Model\RemoteRequestQueryParamsEntityList $queryObject
   *
   * @return \EdgeBox\SyncCore\V2\Raw\Model\RemoteEntityListResponse
   */
  public function getSyncCoreList(Flow $flow, RemoteRequestQueryParamsEntityList $queryObject);

  /**
   * Turn the given entity into a list item to be sent to the Sync Core.
   * If the entity has been deleted, it will be NULL and the data has to be
   * taken from the status entities instead.
   *
   * @param \Drupal\cms_content_sync\Entity\Flow|null $flow
   * @param object|null $entity
   * @param array $statuses
   *
   * @return \EdgeBox\SyncCore\V2\Raw\Model\RemoteEntitySummary
   */
  public function getSyncCoreListItem(?Flow $flow, ?object $entity, array $statuses);

  /**
   * Whether entities of this handler should receive a new UUID when pulled with
   * the Flow update behavior "clone".
   *
   * The decision can depend on the pull mode (automatically/manually/dependency)
   * and the handler/entity type.
   *
   * @param string $pullMode
   *   The pull mode configured for this entity type/bundle.
   *
   * @return bool
   */
  public function shouldClone(string $pullMode);

}
