<?php

namespace Drupal\cms_content_sync\Controller;

use Drupal\cms_content_sync\Entity\Flow;
use Drupal\cms_content_sync\Form\PushChangesConfirm;
use Drupal\cms_content_sync\PushIntent;
use Drupal\cms_content_sync\SyncIntent;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;

/**
 * Push changes controller.
 */
class PushChanges extends ControllerBase {

  /**
   * Immediately push the entity to the Sync Core, without confirmation.
   *
   * @param string $flow_id
   *   The id to the flow that will push to entity.
   * @param \Drupal\Core\Entity\EntityInterface|string $entity
   *   The entity to push.
   * @param string $entity_type
   *   The type of the entity to push.
   *
   * @throws \Exception
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response.
   */
  public static function pushChanges($flow_id, EntityInterface|string $entity, $entity_type = '') {
    if (!$entity instanceof EntityInterface) {
      if ('' == $entity_type) {
        throw new \Exception(t('If no entity object is given, the entity_type is required.'));
      }
      $entity = \Drupal::entityTypeManager()->getStorage($entity_type)->load($entity);
      if (!$entity instanceof EntityInterface) {
        throw new \Exception(t('Entity could not be loaded.'));
      }
    }

    $flow = Flow::load($flow_id);
    if (!PushIntent::pushEntityFromUi(
          $entity,
          PushIntent::PUSH_FORCED,
          SyncIntent::ACTION_UPDATE,
          $flow
      )) {
      $messenger = \Drupal::messenger();
      $messenger->addWarning(t('%label has not been pushed to your @repository: @reason', [
        '%label' => $entity->label(),
        '@repository' => _cms_content_sync_get_repository_name(),
        '@reason' => PushIntent::getNoPushReason($entity, TRUE)
      ]));
    }

    return new RedirectResponse('/');
  }

  /**
   * Confirm pushing multiple entities. This is important to let editors know what
   * other translations they are pushing along as well.
   *
   * @throws \Exception
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   */
  public static function pushMultipleChangesConfirmRedirect() {
    $store = \Drupal::service('tempstore.private')->get('node_cms_content_sync_push_changes_confirm');

    $entities = $store->get('entities') ?? [];

    if (empty($entities)) {
      return new RedirectResponse('/');
    }

    // If we kept this, the user would just be redirected there immediately
    // instead of the new route that we're providing here.
    $options = [];
    $destination = \Drupal::request()->query->get('destination');
    if (isset($destination)) {
      $options = [
        'query' => [
          'destination' => $destination,
        ],
      ];
    }

    \Drupal::request()->query->remove('destination');

    return new RedirectResponse(Url::fromRoute('cms_content_sync.publish_multiple_changes_confirm', [
      'entities' => implode(',', $entities),
    ], $options)->toString());
  }

  /**
   * Confirm pushing this node. This is important to let editors know what
   * other translations they are pushing along as well.
   *
   * @param string $entity
   *
   * @throws \Exception
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   */
  public static function pushChangesConfirm(string $flow_id, string $entity_type, mixed $entity_id, ?string $language) {
    $entity = \Drupal::entityTypeManager()->getStorage($entity_type)->load($entity_id);
    if (!$entity instanceof EntityInterface) {
      throw new \Exception(t('Entity could not be loaded.'));
    }

    if ($language && $entity instanceof TranslatableInterface && $entity->hasTranslation($language)) {
      $entity = $entity->getTranslation($language);
    }

    return \Drupal::formBuilder()->getForm(PushChangesConfirm::class, [$entity], $flow_id);
  }

  /**
   * Confirm pushing this node. This is important to let editors know what
   * other translations they are pushing along as well.
   *
   * @param string $entity
   *
   * @throws \Exception
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   */
  public static function pushChangesConfirmDeletedTranslation(string $flow_id, string $entity_type, mixed $entity_id, ?string $language) {
    $entity = \Drupal::entityTypeManager()->getStorage($entity_type)->load($entity_id);
    if (!$entity instanceof EntityInterface) {
      throw new \Exception(t('Entity could not be loaded.'));
    }

    if ($language && $entity instanceof TranslatableInterface && $entity->hasTranslation($language)) {
      $entity = $entity->getTranslation($language);
    }

    return \Drupal::formBuilder()->getForm(PushChangesConfirm::class, [$entity], $flow_id, TRUE);
  }

  /**
   * Returns an read_list entities for Sync Core.
   *
   * @todo Should be removed when read_list will be allowed to omit.
   */
  public function pushChangesEntitiesList() {
    return new Response('[]');
  }

}
