<?php

namespace Drupal\cms_content_sync\Controller;

use Drupal\cms_content_sync\SyncCoreInterface\SyncCoreFactory;
use Drupal\cms_content_sync\SyncCorePoolExport;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Push changes controller.
 */
class PoolExport extends ControllerBase {

  /**
   * Export pool.
   *
   * @param string $cms_content_sync_pool
   *   The id of the pool.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \EdgeBox\SyncCore\Exception\SyncCoreException
   * @throws \Exception
   *
   * @return null|\Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response.
   */
  public function export(string $cms_content_sync_pool, Request $request): RedirectResponse {
    /**
     * @var \Drupal\cms_content_sync\Entity\Pool $pool
     */
    $pool = \Drupal::entityTypeManager()
      ->getStorage('cms_content_sync_pool')
      ->load($cms_content_sync_pool);
    $messenger = \Drupal::messenger();

    if (!PoolExport::validateBaseUrl($pool)) {
      return new RedirectResponse(Url::fromRoute('entity.cms_content_sync_pool.collection')->toString());
    }

    $exporter = new SyncCorePoolExport($pool);

    $batch = $exporter->prepareBatch(FALSE, !empty($_GET['force']) && 'true' === $_GET['force']);
    $operations = [];
    for ($i = 0; $i < $batch->count(); ++$i) {
      $operations[] = [
            [$batch->get($i), 'execute'],
            [],
      ];
    }

    $batch = [
      'title' => t('Export configuration'),
      'operations' => $operations,
      'finished' => '\Drupal\cms_content_sync\Controller\PoolExport::batchFinished',
    ];
    batch_set($batch);

    return batch_process(Url::fromRoute('entity.cms_content_sync_pool.collection'));
  }

  /**
   * Ensure that the sites base url is valid.
   *
   * @param object $pool
   *   The pool object.
   *
   * @throws \Exception
   *
   * @return bool
   *   Returns true or false
   */
  public static function validateBaseUrl($pool): bool {
    $messenger = \Drupal::messenger();
    $settings = ContentSyncSettings::getInstance();
    $baseUrl = $settings->getSiteBaseUrl(TRUE);
    if (FALSE !== strpos($pool->backend_url ?? SyncCoreFactory::getSyncCoreV2Url(), 'content-sync.io')) {
      if (!UrlHelper::isValid($baseUrl, TRUE)
      || '/' === mb_substr($baseUrl, -1)) {
        $messenger->addMessage(t('The site does not have a valid base url. The base url must be a valid HTTP/HTTPS URL and must not contain a trailing slash. The base url of the site can be configured <a href="@url">here.</a>', [
          '@url' => Url::fromRoute('cms_content_sync.settings_form')->toString(),
        ]), $messenger::TYPE_ERROR);

        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * Batch export finished callback.
   *
   * @param bool $success
   *   Indicate that the batch API tasks were all completed successfully.
   * @param array $results
   *   An array of all the results that were updated.
   * @param array $operations
   *   A list of all the operations that had not been completed by the
   *   batch API.
   */
  public static function batchFinished($success, array $results, array $operations) {
    if ($success) {
      $message = t('Pool has been exported.');
    }
    else {
      $message = t('Pool export failed.');
    }

    \Drupal::messenger()->addMessage($message);
  }

}
