<?php

namespace Drupal\cms_content_sync\Controller;

use Drupal\cms_content_sync\Entity\Pool;
use Drupal\cms_content_sync\SyncCoreInterface\SyncCoreFactory;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Site\Settings;
use EdgeBox\SyncCore\Exception\NotFoundException;
use EdgeBox\SyncCore\Interfaces\IApplicationInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class ContentSyncSettings.
 *
 * Helper functions to set and get the settings of this page.
 */
class ContentSyncSettings {
  /**
   * The Config Factory Interface.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entities to be embedded.
   *
   * @var string[]
   */
  protected $embedEntities;

  /**
   * Option: Preview Enabled.
   *
   * @var bool
   */
  protected $isPreviewEnabled;

  /**
   * Option: Dynamic Pool Assignment.
   *
   * @var bool
   */
  protected $isDynamicPoolAssignmentDisabled;

  /**
   * Option: Authentication type.
   *
   * @var string
   */
  protected $authenticationType;

  /**
   * Config: Site Base Url.
   *
   * @var string
   */
  protected $siteBaseUrl;

  /**
   * Config: Sync Core Url.
   *
   * @var string
   */
  protected $syncCoreUrl;

  /**
   * Config: Site Machine name.
   *
   * @var null|string
   */
  protected $siteMachineName;

  /**
   * Config: Site Id.
   *
   * @var null|string
   */
  protected $siteId;

  /**
   * Config: Site Uuid.
   *
   * @var null|string
   */
  protected $siteUuid;

  /**
   * Config: Site name.
   *
   * @var null|string
   */
  protected $siteName;

  /**
   * Config: Regex pattern for file URLs that should be exported as absolute URLs.
   *
   * @var null|string
   */
  protected $absoluteFileUrlRegex;

  /**
   * Config: Node bundles that should be exported as absolute URLs.
   *
   * @var null|string[]
   */
  protected $absoluteContentUrlBundles;

  /**
   * Constructs a new EntityListBuilder object.
   */
  public function __construct(ConfigFactoryInterface $configFactory) {
    $this->configFactory = $configFactory;
  }

  /**
   * Singleton.
   *
   * @return ContentSyncSettings
   *   Returns the content sync settings.
   */
  public static function getInstance() {
    static $instance = NULL;
    if ($instance) {
      return $instance;
    }

    return $instance = self::createInstance(\Drupal::getContainer());
  }

  /**
   * Create a new instance of the controller with the services of the given container.
   */
  public static function createInstance(ContainerInterface $container) {
    return new static(
    $container->get('config.factory')
    );
  }

  /**
   * Get the embedded entities.
   *
   * @return string[]
   *   Returns the embedded entities.
   */
  public function getEmbedEntities() {
    if (NULL !== $this->embedEntities) {
      return $this->embedEntities;
    }

    $value = $this
      ->configFactory
      ->get('cms_content_sync.settings')
      ->get('cms_content_sync_embed_entities');

    if (!$value) {
      $value = [];
    }

    return $this->embedEntities = $value;
  }

  /**
   * Sets the embed entities.
   *
   * @param string[] $set
   *   The embedded entities.
   */
  public function setEmbedEntities(array $set) {
    $this
      ->configFactory
      ->getEditable('cms_content_sync.settings')
      ->set('cms_content_sync_embed_entities', $set)
      ->save();

    $this->embedEntities = $set;
  }

  /**
   * Option: Preview Enabled.
   *
   * @return bool
   *   Returns true or false if the preview is enabled.
   */
  public function isPreviewEnabled() {
    if (NULL !== $this->isPreviewEnabled) {
      return $this->isPreviewEnabled;
    }

    return $this->isPreviewEnabled = boolval(
          $this
            ->configFactory
            ->get('cms_content_sync.settings')
            ->get('cms_content_sync_enable_preview')
      );
  }

  /**
   * Set preview option.
   *
   * @param bool $set
   *   The value to be set for the preview.
   */
  public function setPreviewEnabled($set) {
    $this
      ->configFactory
      ->getEditable('cms_content_sync.settings')
      ->set('cms_content_sync_enable_preview', $set)
      ->save();

    $this->isPreviewEnabled = $set;
  }

  /**
   * Option: Dynamic Pool Assignment.
   *
   * @return bool
   *   Returns true or false if the dynamic pool assignment is enabled.
   */
  public function isDynamicPoolAssignmentDisabled() {
    if (NULL !== $this->isDynamicPoolAssignmentDisabled) {
      return $this->isDynamicPoolAssignmentDisabled;
    }

    return $this->isDynamicPoolAssignmentDisabled = boolval(
          $this
            ->configFactory
            ->get('cms_content_sync.settings')
            ->get('cms_content_sync_disable_dynamic_pool_assignment')
      );
  }

  /**
   * Set dynamic pool assignment option.
   *
   * @param bool $set
   *   The value to be set for the dynamic pool assignment.
   */
  public function disableDynamicPoolAssignment($set) {
    $this
      ->configFactory
      ->getEditable('cms_content_sync.settings')
      ->set('cms_content_sync_disable_dynamic_pool_assignment', $set)
      ->save();

    $this->isDynamicPoolAssignmentDisabled = $set;
  }

  /**
   * Get the authentication type.
   *
   * @return string
   *   The authentication type.
   */
  public function getAuthenticationType() {
    if (NULL !== $this->authenticationType) {
      return $this->authenticationType;
    }

    $this->authenticationType = $this
      ->configFactory
      ->get('cms_content_sync.settings')
      ->get('cms_content_sync_authentication_type');

    if (!$this->authenticationType) {
      if (\Drupal::service('module_handler')->moduleExists('basic_auth')) {
        $this->authenticationType = IApplicationInterface::AUTHENTICATION_TYPE_BASIC_AUTH;
      }
      else {
        $this->authenticationType = IApplicationInterface::AUTHENTICATION_TYPE_COOKIE;
      }
    }

    return $this->authenticationType;
  }

  /**
   * Set the authentication type.
   *
   * @param string $set
   *   The value to be set for the authentication type.
   */
  public function setAuthenticationType($set) {
    $this
      ->configFactory
      ->getEditable('cms_content_sync.settings')
      ->set('cms_content_sync_authentication_type', $set)
      ->save();

    $this->authenticationType = $set;
  }

  /**
   * Get the regex pattern for file URLs that should be exported as absolute URLs.
   *
   * @return null|string
   *   The regex pattern without delimiters, or NULL if not configured.
   */
  public function getAbsoluteFileUrlRegex() {
    if (NULL !== $this->absoluteFileUrlRegex) {
      return $this->absoluteFileUrlRegex;
    }

    $value = $this
      ->configFactory
      ->get('cms_content_sync.settings')
      ->get('cms_content_sync_absolute_file_url_regex');

    $value = is_string($value) ? trim($value) : '';

    return $this->absoluteFileUrlRegex = ($value === '' ? NULL : $value);
  }

  /**
   * Set the regex pattern for file URLs that should be exported as absolute URLs.
   *
   * @param null|string $set
   *   The regex pattern without delimiters.
   */
  public function setAbsoluteFileUrlRegex($set) {
    $set = is_string($set) ? trim($set) : '';

    $this
      ->configFactory
      ->getEditable('cms_content_sync.settings')
      ->set('cms_content_sync_absolute_file_url_regex', $set)
      ->save();

    $this->absoluteFileUrlRegex = ($set === '' ? NULL : $set);
  }

  /**
   * Get the node bundles that should be exported as absolute URLs.
   *
   * @return string[]
   *   A list of node bundle machine names.
   */
  public function getAbsoluteContentUrlBundles() {
    if (NULL !== $this->absoluteContentUrlBundles) {
      return $this->absoluteContentUrlBundles;
    }

    $value = $this
      ->configFactory
      ->get('cms_content_sync.settings')
      ->get('cms_content_sync_absolute_content_url_bundles');

    if (!is_array($value)) {
      $value = [];
    }

    $value = array_values(array_filter($value, fn($bundle) => is_string($bundle) && $bundle !== ''));

    return $this->absoluteContentUrlBundles = $value;
  }

  /**
   * Set the node bundles that should be exported as absolute URLs.
   *
   * @param string[] $set
   *   The bundle machine names.
   */
  public function setAbsoluteContentUrlBundles(array $set) {
    $set = array_values(array_filter($set, fn($bundle) => is_string($bundle) && $bundle !== ''));

    $this
      ->configFactory
      ->getEditable('cms_content_sync.settings')
      ->set('cms_content_sync_absolute_content_url_bundles', $set)
      ->save();

    $this->absoluteContentUrlBundles = $set;
  }

  /**
   * Get the sites base url.
   *
   * @param bool $config_only
   *   Whether to return the base URL as per the settings only.
   *   Otherwise it will return the default site URL.
   *
   * @throws \Exception
   *
   * @return string
   *   The base URL.
   */
  public function getSiteBaseUrl($config_only = FALSE) {
    if (NULL !== $this->siteBaseUrl) {
      return $this->siteBaseUrl;
    }

    // Allow overwritting of the base_url from the global $settings.
    $setting = Settings::get('cms_content_sync_base_url', \Drupal::state()->get('cms_content_sync.base_url'));

    if ($config_only) {
      return $setting;
    }

    if ($setting) {
      // Validate the Base URL.
      if (UrlHelper::isValid($setting, TRUE) && '/' !== mb_substr($setting, -1)) {
        return $this->siteBaseUrl = $setting;
      }

      throw new \Exception(t('The configured base URL is not a valid URL. Ensure that it does not contain a trailing slash.'));
    }

    global $base_url;

    return $this->siteBaseUrl = $base_url;
  }

  /**
   * Set the site base URL.
   *
   * @param string $set
   *   The base URL to be set.
   */
  public function setSiteBaseUrl($set) {
    \Drupal::state()->set('cms_content_sync.base_url', $set);

    // Reset so it's computed again.
    $this->siteBaseUrl = NULL;
  }

  /**
   * Get the sync core Url from the configuration.
   *
   * @return null|string
   *   The sync core URL.
   */
  public function getSyncCoreUrl() {
    if (NULL !== $this->syncCoreUrl) {
      return $this->syncCoreUrl;
    }

    $default = $this
      ->configFactory
      ->get('cms_content_sync.settings')
      ->get('cms_content_sync_sync_core_url');

    return $this->syncCoreUrl = Settings::get('cms_content_sync_sync_core_url', $default);
  }

  /**
   * Set the sync core URL.
   *
   * @param string $set
   *   The sync core Url value to be set.
   */
  public function setSyncCoreUrl($set) {
    $this
      ->configFactory
      ->getEditable('cms_content_sync.settings')
      ->set('cms_content_sync_sync_core_url', $set)
      ->save();

    // Reset so it's computed again.
    $this->syncCoreUrl = NULL;
  }

  /**
   * Option: Direct Sync Core Access.
   *
   * Whether or not users can directly communicate with the Sync Core. This only makes sense if the Sync Core is
   * exposed publicly so right it's restricted to the Cloud version of Content Sync.
   * The setting is saved in the Sync Core, not locally at this site. Otherwise the configuration of multiple sites
   * would conflict and lead to undesired outcomes.
   *
   * @param bool $default
   *   What to assume as the default value if no explicit settings are saved yet.
   *
   * @throws \EdgeBox\SyncCore\Exception\SyncCoreException
   *
   * @return bool
   *   Retuns true if direct sync core access is enabled.
   */
  public function isDirectSyncCoreAccessEnabled($default = TRUE) {
    static $cache = NULL;

    if (NULL !== $cache) {
      return $cache;
    }

    $checked = [];

    // We don't distinguish between multiple Sync Cores. If it's enabled for one, it's enabled for all.
    foreach (Pool::getAll() as $pool) {
      $url = $pool->getSyncCoreUrl();
      if (in_array($url, $checked)) {
        continue;
      }
      $checked[] = $url;

      try {
        $setting = $pool
          ->getClient()
          ->isDirectUserAccessEnabled();

        if ($setting) {
          return $cache = TRUE;
        }
        if (FALSE === $setting && NULL === $cache) {
          $cache = FALSE;
        }
      }
      catch (NotFoundException $e) {
        // No config exported yet, so skip this.
        continue;
      }
    }

    // If it's not set at all, we default it to TRUE so people benefit from the increased performance.
    if (NULL === $cache) {
      return $cache = $default;
    }

    return $cache = FALSE;
  }

  /**
   * Set whether or not to allow direct Sync Core communication.
   *
   * @param bool $set
   *   The value to be set.
   *
   * @throws \Exception
   */
  public function setDirectSyncCoreAccessEnabled($set) {
    $checked = [];

    // We don't distinguish between multiple Sync Cores. If it's enabled for one, it's enabled for all.
    foreach (Pool::getAll() as $pool) {
      $url = $pool->getSyncCoreUrl();
      if (in_array($url, $checked)) {
        continue;
      }
      $checked[] = $url;

      try {
        $pool
          ->getClient()
          ->isDirectUserAccessEnabled($set);
      }
      catch (NotFoundException $e) {
        // No config exported yet, so skip this.
        continue;
      }
    }
  }

  /**
   * Returns the site machine name.
   *
   * @return null|string
   *   Returns the site machine name.
   */
  public function getSiteMachineName() {
    if (NULL !== $this->siteMachineName) {
      return $this->siteMachineName;
    }

    return $this->siteId = Settings::get('cms_content_sync_site_machine_name', \Drupal::state()->get('cms_content_sync.site_machine_name'));
  }

  /**
   * Sets the site machine name.
   *
   * @param string $set
   *   The site machine name to be set.
   */
  public function setSiteMachineName($set) {
    \Drupal::state()->set('cms_content_sync.site_machine_name', $set);

    // Reset so it's computed again.
    $this->siteMachineName = NULL;
  }

  /**
   * Get the site Id.
   *
   * @throws \Exception
   *
   * @return null|string
   *   Returns the site Id.
   */
  public function getSiteId() {
    if (NULL !== $this->siteId) {
      return $this->siteId;
    }

    // Allow overwritting of the site_id from the global $settings.
    return $this->siteId = Settings::get('cms_content_sync_site_id', \Drupal::state()->get('cms_content_sync.site_id'));
  }

  /**
   * Set the site Id.
   *
   * @param string $set
   *   The site Id to be set.
   */
  public function setSiteId($set) {
    \Drupal::state()->set('cms_content_sync.site_id', $set);

    // Reset so it's computed again.
    $this->siteId = NULL;
  }

  /**
   * Get the sites Uuid.
   *
   * @throws \Exception
   *
   * @return null|string
   *   Returns the sites Uuid.
   */
  public function getSiteUuid() {
    if (NULL !== $this->siteUuid) {
      return $this->siteUuid;
    }

    // Allow overwritting of the site_uuid from the global $settings.
    return $this->siteUuid = Settings::get('cms_content_sync_site_uuid', \Drupal::state()->get('cms_content_sync.site_uuid'));
  }

  /**
   * Sets the sites Uuid.
   *
   * @param string $set
   *   The sites Uuid to be set.
   */
  public function setSiteUuid($set) {
    \Drupal::state()->set('cms_content_sync.site_uuid', $set);

    // Reset so it's computed again.
    $this->siteUuid = NULL;
  }

  /**
   * Reset the site Id.
   *
   * Remove the site ID setting from the current site's state, requesting a
   * new one during the next config export.
   */
  public function resetSiteId() {
    // Reset site ID.
    \Drupal::state()->delete('cms_content_sync.site_id');
    // Reset so it's computed again.
    $this->siteId = NULL;

    // Reset machine name.
    \Drupal::state()->delete('cms_content_sync.site_machine_name');
    // Reset so it's computed again.
    $this->siteMachineName = NULL;
  }

  /**
   * Gets the site Name.
   *
   * @return null|string
   *   Returns the site name.
   */
  public function getSiteName() {
    if (NULL !== $this->siteName) {
      return $this->siteName;
    }

    // Allow overwritting of the site_name from the global $settings.
    if (Settings::get('cms_content_sync_site_name')) {
      return $this->siteName = Settings::get('cms_content_sync_site_name');
    }

    // If we aren't connected yet, don't ask the Sync Core for our name (will
    // throw an exception).
    if ($this->getSiteId()) {
      $core = SyncCoreFactory::getAnySyncCore();
      if ($core) {
        $stored = $core->getSiteName();

        if ($stored) {
          return $this->siteName = $stored;
        }
      }
    }

    $this->siteName = $this
      ->configFactory
      ->get('system.site')
      ->get('name');

    return $this->siteName;
  }

  /**
   * Sets the site name.
   *
   * @param string $set
   *   The site name to be set.
   */
  public function setSiteName($set) {
    foreach (SyncCoreFactory::getAllSyncCores() as $core) {
      $core->setSiteName($set);
    }
  }

  /**
   * Get the Extended Entity Export Logging setting.
   */
  public function getExtendedEntityExportLogging() {
    return \Drupal::service('keyvalue.database')->get('cms_content_sync_debug')->get('extended_entity_export_logging');
  }

  /**
   * Set the Extended Entity Export Logging setting.
   *
   * @param mixed $value
   *   The value to be set for the extended entity export logging setting.
   */
  public function setExtendedEntityExportLogging($value) {
    $key_value_db = \Drupal::service('keyvalue.database');
    $key_value_db->get('cms_content_sync_debug')->set('extended_entity_export_logging', $value);
  }

  /**
   * Get the Extended Entity Import Logging setting.
   */
  public function getExtendedEntityImportLogging() {
    return \Drupal::service('keyvalue.database')->get('cms_content_sync_debug')->get('extended_entity_import_logging');
  }

  /**
   * Set the Extended Entity Import Logging setting.
   *
   * @param mixed $value
   *   The value to be set for the extended entity export logging setting.
   */
  public function setExtendedEntityImportLogging($value) {
    $key_value_db = \Drupal::service('keyvalue.database');

    $key_value_db->get('cms_content_sync_debug')->set('extended_entity_import_logging', $value);
  }

}
