<?php

namespace Drupal\cms_content_sync\Form;

use Drupal\cms_content_sync\Entity\EntityStatus;
use Drupal\cms_content_sync\Entity\Flow;
use Drupal\cms_content_sync\Plugin\Type\EntityHandlerPluginManager;
use Drupal\cms_content_sync\PushIntent;
use Drupal\cms_content_sync\SyncCoreInterface\SyncCoreFactory;
use Drupal\cms_content_sync\SyncIntent;
use Drupal\Core\Entity\EntityChangedInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManager;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Provides a node deletion confirmation form.
 *
 * @internal
 */
class PushChangesConfirm extends ConfirmFormBase {
  /**
   * The tempstore factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStoreFactory;

  /**
   * The node storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $storage;

  /**
   * The selected nodes to push.
   *
   * @var \Drupal\Core\Entity\EntityInterface[]
   */
  protected $entities;

  /**
   * The Flow to use for pushing, if any.
   *
   * @var null|Flow
   */
  protected $flow;

  /**
   * The entities in their default language along with all the translations to be pushed.
   *
   * @var array
   */
  protected $tree;

  /**
   * Whether any of the selected entities has additional translations, meaning
   * we show a more complex table.
   *
   * @var bool
   */
  protected $hasTranslations = FALSE;

  /**
   * Constructs a DeleteMultiple form object.
   *
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The tempstore factory.
   * @param \Drupal\Core\Entity\EntityTypeManager $manager
   *   The entity manager.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function __construct(PrivateTempStoreFactory $temp_store_factory, EntityTypeManager $manager) {
    $this->tempStoreFactory = $temp_store_factory;
    $this->storage = $manager->getStorage('node');
    $this->entities = $this->tempStoreFactory->get('node_cms_content_sync_push_changes_confirm')->get('entities');
    $flow_id = $this->tempStoreFactory->get('node_cms_content_sync_push_changes_confirm')->get('flow_id');
    $this->flow = $flow_id ? Flow::getAll()[$flow_id] : NULL;

    $this->tree = [];

    foreach ($this->entities as $entity) {
      foreach ($this->tree as $item) {
        if ($item['root']->id() === $entity->id()) {
          continue 2;
        }
      }

      $root_entity = $entity instanceof TranslatableInterface ? $entity->getUntranslated() : $entity;

      $item = [
        'root' => $root_entity,
        'translations' => [],
        'root_selected' => in_array($root_entity, $this->entities),
        'root_required' => !EntityStatus::getLastPushForEntity($root_entity),
        'translations_selected' => [],
        'pushed_translations' => $entity instanceof TranslatableInterface ? SyncCoreFactory::getSyncCoreV2()->getUsedLanguages($entity->getEntityTypeId(), $entity->bundle(), EntityHandlerPluginManager::getSharedId($entity)) : NULL,
      ];

      if ($root_entity instanceof TranslatableInterface && count($root_entity->getTranslationLanguages()) > 1) {
        $languages = $root_entity->getTranslationLanguages();
        foreach ($languages as $id => $language) {
          $translation = $root_entity->getTranslation($id);

          $translation_id = $this->getEntityLanguageId($translation);
          if ($id !== $root_entity->language()->getId()) {

            if (in_array($translation, $this->entities)) {
              $item['translations_selected'][] = $translation_id;
            }
          }

          $item['translations'][$id] = $translation;
        }
      }

      $this->tree[] = $item;
    }

    foreach ($this->tree as $tree_item) {
      if (!empty($tree_item['translations'])) {
        $this->hasTranslations = TRUE;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
    $container->get('tempstore.private'),
    $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'node_cms_content_sync_push_changes_confirm';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return 'Please confirm pushing';
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('system.admin_content');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return _cms_content_sync_push_label();
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    if (empty($this->entities)) {
      return new RedirectResponse($this->getCancelUrl()->setAbsolute()->toString());
    }

    $show_order = !!$form_state->getValue('show-order');

    if ($show_order) {
      \Drupal::messenger()->addWarning($this->t('When defining a push order, the content is pushed and pulled one after another. This is useful when:<ol><li>The publishing is time-sensitive and specific content should be published first.</li><li>You have references to new content that you want to be resolved immediately.</li></ol><br/><strong>But if any push or pull fails, the push/pull of every content item after it will fail, too! The total time to update content is also considerably longer.</strong>'));
    }

    $can_select_translations = PushIntent::canHandleTranslationsIndependently();

    $items = [
      '#type' => 'table',
      '#attributes' => [
        'id' => 'content-sync--push-changes-confirm',
      ],
      /*'#header' => [
        $show_order ? $this->t('Order') : '',
        $this->t('Title'),
      ],*/
      '#tabledrag' => $show_order ? [[
        'action' => 'order',
        'relationship' => 'sibling',
        'group' => 'draggable-weight',
      ]
] : [],
    ];

    foreach ($this->tree as $weight => $tree_item) {
      $entity = $tree_item['root'];
      $pushed_translations = $tree_item['pushed_translations'];
      $root_selected = $tree_item['root_selected'];
      if ($show_order) {
        $items[$entity->id()]['#attributes']['class'] = ['draggable'];
        $items[$entity->id()]['#weight'] = $weight;
      }

      if ($show_order) {
        $items[$entity->id()]['order-handle'] = [
          '#markup' => '',
        ];
        $items[$entity->id()]['order'] = [
          '#type' => 'weight',
          '#title' => t('Weight'),
          '#title_display' => 'invisible',
          '#default_value' => $weight,
          '#attributes' => [
            'class' => [
              'draggable-weight',
            ],
          ],
        ];
      }

      if ($entity instanceof TranslatableInterface && $this->hasTranslations) {
        $items[$entity->id()]['content'] = [
          'title' => [
            '#markup' => $entity->label(),
            '#prefix' => '<h1>',
            '#suffix' => '</h1>',
          ],
        ];

        $translations = [
          '#type' => 'table',
          '#header' => [
            $this->t('Push'),
            $this->t('Language'),
            $this->t('Updated'),
            $this->t('Last pushed version'),
            $this->t('Notes'),
            $this->t('Title'),
          ],
          '#attributes' => [
            'class' => [
              'pushable-translations',
            ],
          ],
        ];

        $last_push = !$pushed_translations || in_array($entity->language()->getId(), $pushed_translations) ? EntityStatus::getLastPushForEntity($entity) : NULL;
        $changed_time = $entity instanceof EntityChangedInterface ? $entity->getChangedTime() : NULL;

        $id = $entity->language()->getId();
        $has_changed = !$last_push || !$changed_time || $changed_time > $last_push;
        $default_value = $root_selected || $tree_item['root_required'] || !$can_select_translations;
        $translations[$id]['#attributes']['class'] = [
          $last_push ? ($has_changed ? 'entity-changed' : 'entity-unchanged') : 'entity-new',
          $default_value ? 'default-checked' : 'default-unchecked',
        ];
        $translations[$id]['push'] = [
          '#type' => 'checkbox',
          '#default_value' => $default_value,
          '#disabled' => $tree_item['root_required'] || !$can_select_translations,
          '#title_display' => 'invisible',
          '#title' => $this->t('Push'),
          '#attributes' => [
            'title' => $tree_item['root_required'] ? $this->t('The default language of a content must always be pushed initially as it cannot be changed later; this is a technical requirement from Drupal.') : (!$can_select_translations ? $this->t('Please enable individual translation handling to use this (see below).') : ''),
          ],
        ];
        $translations[$id]['language'] = [
          '#markup' => $entity->language()->getName(),
        ];
        $translations[$id]['changed'] = $changed_time ? [
          '#markup' => \Drupal::service('date.formatter')->format($changed_time),
        ] : [
          '#markup' => '',
        ];
        $translations[$id]['pushed'] = $last_push ? [
          '#markup' => \Drupal::service('date.formatter')->format($last_push),
        ] : [
          '#prefix' => '<em>',
          '#suffix' => '<em>',
          '#markup' => $this->t('New'),
        ];
        $translations[$id]['notes'] = [
          '#markup' => ($tree_item['root_required'] ? '<em>' . $this->t('default') . '</em>, <strong>' . $this->t('required') . '</strong>' : '<em>' . $this->t('default') . '</em>') . ($has_changed ? '' : ', <em>' . $this->t('unchanged') . '</em>'),
        ];
        $translations[$id]['title'] = [
          '#markup' => $entity->label(),
        ];

        foreach ($tree_item['translations'] as $translation) {
          $translation_id = $translation->language()->getId();
          if ($translation_id === $id) {
            continue;
          }
          $changed_time = $translation instanceof EntityChangedInterface ? $translation->getChangedTime() : NULL;
          $last_push = !$pushed_translations || in_array($translation->language()->getId(), $pushed_translations) ? EntityStatus::getLastPushForEntity($translation) : NULL;
          $has_changed = !$last_push || !$changed_time || $changed_time > $last_push;
          $default_value = in_array($this->getEntityLanguageId($translation), $tree_item['translations_selected']) || !$can_select_translations;
          $translations[$translation_id]['#attributes']['class'] = [
            $last_push ? ($has_changed ? 'entity-changed' : 'entity-unchanged') : 'entity-new',
            $default_value ? 'default-checked' : 'default-unchecked',
          ];
          $translations[$translation_id]['push'] = [
            '#type' => 'checkbox',
            '#default_value' => $default_value,
            '#disabled' => !$can_select_translations,
            '#title' => $this->t('Push'),
            '#title_display' => 'invisible',
            '#attributes' => [
              'class' => [
                'push-select',
              ],
            ],
          ];
          $translations[$translation_id]['language'] = [
            '#markup' => $translation->language()->getName(),
          ];
          $translations[$translation_id]['changed'] = $changed_time ? [
            '#markup' => \Drupal::service('date.formatter')->format($changed_time),
          ] : [
            '#markup' => '',
          ];
          $translations[$translation_id]['pushed'] = $last_push ? [
            '#markup' => \Drupal::service('date.formatter')->format($last_push),
          ] : [
            '#prefix' => '<em>',
            '#suffix' => '<em>',
            '#markup' => $this->t('New'),
          ];
          $translations[$translation_id]['notes'] = [
            '#markup' => $has_changed ? '' : $this->t('unchanged'),
          ];
          $translations[$translation_id]['title'] = [
            '#markup' => $translation->label(),
          ];
        }

        if (count($translations) > 1 && $can_select_translations) {
          $translations['actions']['actions']['#wrapper_attributes']['colspan'] = 5;
          $translations['actions']['actions']['#wrapper_attributes']['class'] = ['entity-actions'];
          $translations['actions']['actions']['#markup'] = $this->t('Select') . ': <a href="#" class="select-all-action">' . $this->t('all') . '</a> | <a href="#" class="select-new-action">' . $this->t('new') . '</a> | <a href="#" class="select-new-changed-action">' . $this->t('new and changed') . '</a> | <a href="#" class="select-changed-action">' . $this->t('changed') . '</a> | <a href="#" class="select-none-action">' . $this->t('none') . '</a> | <a href="#" class="select-restore-action">' . $this->t('restore') . '</a>';
        }

        $items[$entity->id()]['content']['translations'] = $translations;
      }
      else {
        $items[$entity->id()]['content'] = [
          'push' => [
            '#type' => 'checkbox',
            '#default_value' => $root_selected || $tree_item['root_required'],
            '#disabled' => $tree_item['root_required'],
            '#title' => $entity->label(),
          ],
          '#prefix' => '<h1>',
          '#suffix' => '</h1>',
        ];
      }
    }
    $form['entities'] = $items;

    $form = parent::buildForm($form, $form_state);

    if ($show_order) {
      $form['show-order'] = [
        '#type' => 'hidden',
        '#value' => 1,
      ];
    }
    elseif (count($this->tree) > 1) {
      $form['actions']['show-order'] = [
        '#type' => 'submit',
        '#value' => $this->t('Select order'),
        '#submit' => ['::showOrder'],
      ];
    }

    // Swap order.
    $cancel = $form['actions']['cancel'];
    unset($form['actions']['cancel']);
    $form['actions']['cancel'] = $cancel;

    $form['#attached']['library'][] = 'cms_content_sync/push-changes-confirm-form';

    return $form;
  }

  /**
   *
   */
  public function showOrder(array &$form, FormStateInterface $form_state) {
    $form_state->set('show-order', TRUE)->setRebuild(TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    if ($form_state->getValue('confirm')) {
      $ordered = !!$form_state->getValue('show-order');

      if ($ordered) {
        usort($this->tree, function ($a, $b) use (&$form_state) {
          $weight_a = $form_state->getValue(['entities', $a['root']->id(), 'order']);
          $weight_b = $form_state->getValue(['entities', $b['root']->id(), 'order']);
          return $weight_a - $weight_b;
        });
      }

      $translation_groups_per_flow = [];
      foreach ($this->tree as $item) {
        $entity = $item['root'];
        $flow = $this->flow ?? PushIntent::getFlowsForEntity($entity, PushIntent::PUSH_FORCED, SyncIntent::ACTION_CREATE)[0];
        if (!$flow) {
          continue;
        }

        if (!($entity instanceof TranslatableInterface) || !$this->hasTranslations) {
          $selected = !!$form_state->getValue(['entities', $entity->id(), 'content', 'push']);
          if ($selected) {
            $translation_groups_per_flow[$flow->id][] = [$entity];
          }
        }
        else {
          $selected_translations = [];
          $translations = empty($item['translations']) ? [$entity] : $item['translations'];
          if (!in_array($entity, $translations)) {
            $translations[] = $entity;
          }

          foreach ($translations as $translation) {
            $translation_id = $translation->language()->getId();
            $selected = !!$form_state->getValue(['entities', $translation->id(), 'content', 'translations', $translation_id, 'push']);
            if ($selected) {
              $selected_translations[] = $translation;
            }
          }

          if (!empty($selected_translations)) {
            $translation_groups_per_flow[$flow->id][] = $selected_translations;
          }
        }
      }

      foreach ($translation_groups_per_flow as $flow_id => $translation_groups) {
        $flow = Flow::getAll()[$flow_id];
        PushIntent::pushEntitiesFromUi($translation_groups, PushIntent::PUSH_MANUALLY, SyncIntent::ACTION_CREATE, $flow, $ordered);
      }

      $this->tempStoreFactory->get('node_cms_content_sync_push_changes_confirm')->delete('entities');
      $this->tempStoreFactory->get('node_cms_content_sync_push_changes_confirm')->delete('flow_id');
    }

    $form_state->setRedirect('system.admin_content');
  }

  /**
   *
   */
  public function getDescription() {
    $can_select_translations = PushIntent::canHandleTranslationsIndependently();
    if ($can_select_translations) {
      return $this->t('This action cannot be undone.');
    }
    elseif (Url::fromRoute('cms_content_sync.advanced')->access(\Drupal::currentUser())) {
      return $this->t('This action cannot be undone. If you want to publish translations individually, please enable both the <em>Individual requests per translation</em> and the <em>Skip unchanged translations</em> settings in the @advanced_settings_tab tab of the module.', [
        '@advanced_settings_tab' => Link::createFromRoute(t('Advanced settings tab'), 'cms_content_sync.advanced')->toString(),
      ]);
    }
    else {
      return $this->t('This action cannot be undone. If you want to publish translations individually, please ask your site builder to enable the option at the Advanced settings tab of the module.');
    }
  }

  /**
   *
   */
  protected function getEntityLanguageId($entity) {
    return $entity instanceof TranslatableInterface ? $entity->getEntityTypeId() . ":" . $entity->id() . ":" . $entity->language()->getId() : $entity->getEntityTypeId() . ":" . $entity->id();
  }

}
