<?php

namespace Drupal\cms_content_sync;

/**
 * Provides class for groups of status entities that span multiple pools.
 * Required to efficiently push/pull content that has multiple pools assigned.
 *
 * @ingroup cms_content_sync_entity_status
 */
class EntityStatusProxy implements EntityStatusProxyInterface {
  /**
   * @var EntityStatusProxyInterface[]
   */
  protected $statusEntities;

  /**
   * @param EntityStatusProxyInterface[] $status_entities
   */
  public function __construct(array $status_entities) {
    $this->statusEntities = $status_entities;
  }

  /**
   * @inheritDoc
   */
  public function resetStatus() {
    $this->proxyAll('resetStatus');
  }

  /**
   * @inheritDoc
   */
  public function getEntity() {
    return $this->proxyAny('getEntity');
  }

  /**
   * @inheritDoc
   */
  public function wasLastPushReset($set = NULL) {
    return $this->proxyAll('wasLastPushReset', [$set], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function wasLastPullReset($set = NULL) {
    return $this->proxyAll('wasLastPullReset', [$set], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function didPushFail($set = NULL, $soft = FALSE, $details = NULL) {
    return $this->proxyAll('didPushFail', [$set, $soft, $details], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function whyDidPushingFail() {
    return $this->proxyAny('whyDidPushingFail', []);
  }

  /**
   * @inheritDoc
   */
  public function didPullFail($set = NULL, $soft = FALSE, $details = NULL) {
    return $this->proxyAll('didPullFail', [$set, $soft, $details], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function whyDidPullingFail() {
    return $this->proxyAny('whyDidPullingFail', []);
  }

  /**
   * @inheritDoc
   */
  public function isPushEnabled($set = NULL, $setDependency = NULL) {
    return $this->proxyAll('isPushEnabled', [$set, $setDependency], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function isManualPushEnabled() {
    return $this->proxyAny('isManualPushEnabled', []);
  }

  /**
   * @inheritDoc
   */
  public function isPushedAsDependency() {
    return $this->proxyAny('isPushedAsDependency', []);
  }

  /**
   * @inheritDoc
   */
  public function isOverriddenLocally($set = NULL, $individual = FALSE) {
    return $this->proxyAll('isOverriddenLocally', [$set, $individual], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function isSourceEntity($set = NULL, $individual = FALSE) {
    return $this->proxyAll('isSourceEntity', [$set, $individual], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function didUserEnablePush($set = NULL) {
    return $this->proxyAll('didUserEnablePush', [$set], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function isDeleted($set = NULL) {
    return $this->proxyAll('isDeleted', [$set], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function wasPushedEmbedded($set = NULL) {
    return $this->proxyAll('wasPushedEmbedded', [$set], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function wasPulledEmbedded($set = NULL) {
    return $this->proxyAll('wasPulledEmbedded', [$set], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function setParentEntity($type, $uuid) {
    return $this->proxyAll('setParentEntity', [$type, $uuid], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function getParentEntity() {
    return $this->proxyAny('getParentEntity');
  }

  /**
   * @inheritDoc
   */
  public function getLastPull(?string $language = NULL) {
    return $this->proxyAny('getLastPull', [$language]);
  }

  /**
   * @inheritDoc
   */
  public function setLastPull($timestamp, ?string $language = NULL) {
    return $this->proxyAll('setLastPull', [$timestamp, $language], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function getUuid() {
    return $this->proxyAny('getUuid');
  }

  /**
   * @inheritDoc
   */
  public function getEntityTypeName() {
    return $this->proxyAny('getEntityTypeName');
  }

  /**
   * @inheritDoc
   */
  public function getLastPush(?string $language = NULL) {
    return $this->proxyAny('getLastPush', [$language]);
  }

  /**
   * @inheritDoc
   */
  public function setLastPush($timestamp, ?string $language = NULL) {
    return $this->proxyAll('setLastPush', [$timestamp, $language], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function getLastPushTrigger(?string $language = NULL) {
    return $this->proxyAny('getLastPushTrigger', [$language]);
  }

  /**
   * @inheritDoc
   */
  public function setLastPushTrigger(?int $timestamp, ?string $language = NULL) {
    return $this->proxyAll('setLastPushTrigger', [$timestamp, $language], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function getFlow() {
    return $this->proxyAny('getFlow');
  }

  /**
   * @inheritDoc
   */
  public function getEntityTypeVersion() {
    return $this->proxyAny('getEntityTypeVersion');
  }

  /**
   * @inheritDoc
   */
  public function setEntityTypeVersion($version) {
    return $this->proxyAll('setEntityTypeVersion', [$version], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function getSourceUrl() {
    return $this->proxyAny('getSourceUrl');
  }

  /**
   * @inheritDoc
   */
  public function setSourceUrl($url) {
    $this->proxyAll('setSourceUrl', [$url]);
  }

  /**
   * @inheritDoc
   */
  public function getData($key = NULL) {
    return $this->proxyAny('getData', [$key]);
  }

  /**
   * @inheritDoc
   */
  public function setData($key, $value) {
    return $this->proxyAll('setData', [$key, $value], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function getEntityPushHash() {
    return $this->proxyAny('getEntityPushHash');
  }

  /**
   * @inheritDoc
   */
  public function setEntityPushHash($hash) {
    return $this->proxyAll('setEntityPushHash', [$hash], TRUE);
  }

  /**
   * @inheritDoc
   */
  public function save() {
    $this->proxyAll('save');
  }

  /**
   * @inheritDoc
   */
  public function getAllTranslationSourceUrls() {
    $result = $this->proxyAll('getAllTranslationSourceUrls');

    return array_reduce($result, function ($a, $b) {
        return empty($b) ? $a : array_merge($a, $b);
    }, []);
  }

  /**
   * @inheritDoc
   */
  public function getTranslationSourceUrl(string $language, $return_default_if_null = TRUE) {
    $url = $this->proxyAny('getTranslationSourceUrl', [$language, FALSE]);
    if ($url) {
      return $url;
    }
    if ($return_default_if_null) {
      return $this->getSourceUrl();
    }

    return NULL;
  }

  /**
   * @inheritDoc
   */
  public function setTranslationSourceUrl(string $language, string $url) {
    $this->proxyAll('setTranslationSourceUrl', [$language, $url]);
  }

  /**
   * @inheritDoc
   */
  public function getPushedLanguages() {
    $all = $this->proxyAll('getPushedLanguages');
    $languages = [];
    $has_any = FALSE;
    foreach ($all as $result) {
      if (!$result) {
        continue;
      }
      $has_any = TRUE;
      foreach ($result as $language) {
        if (!in_array($language, $languages)) {
          $languages[] = $language;
        }
      }
    }
    if (!$has_any) {
      return NULL;
    }
    return $languages;
  }

  /**
   *
   */
  protected function proxyAll($method, $args = [], $returnSingle = FALSE) {
    $result = [];
    foreach ($this->statusEntities as $status_entity) {
      $result[] = call_user_func_array([$status_entity, $method], $args);
    }

    return $returnSingle
        ? array_reduce($result, function ($a, $b) {
            return empty($a) ? $b : $a;
        })
        : $result;
  }

  /**
   *
   */
  protected function proxyAny($method, $args = []) {
    $result = NULL;
    foreach ($this->statusEntities as $status_entity) {
      $result = call_user_func_array([$status_entity, $method], $args);
      if (!empty($result)) {
        return $result;
      }
    }

    return $result;
  }

}
