<?php

namespace Drupal\cms_content_sync_private_environment\Commands;

use Drupal\cms_content_sync\Cli\CliService;
use Drupal\cms_content_sync_private_environment\Controller\RequestHandlerController;
use Drush\Commands\DrushCommands;

/**
 * Content Sync Drush Commands.
 */
class CMSContentSyncPrivateEnvironmentCommands extends DrushCommands {
  /**
   * The interoperability cli service.
   *
   * @var \Drupal\cms_content_sync\Cli\CliService
   */
  protected $cliService;

  /**
   * CMS Content Sync constructor.
   *
   * @param \Drupal\cms_content_sync\Cli\CliService $cliService
   *   The CLI service which allows interoperability.
   */
  public function __construct(CliService $cliService) {
    $this->cliService = $cliService;

    parent::__construct();
  }

  /**
   * Handle requests from the Content Sync service asynchronously. Requires
   * Basic Auth to be enabled and configured for Content Sync.
   *
   * @param string $limit
   *   Either "watch", "all" or a fixed number.
   * @param array $options
   *   An array containing the option parameters provided by Drush.
   *
   * @command cms_content_sync_private_environment:poll
   *
   * @aliases cspep
   *
   * @options pollInterval
   *  The interval in seconds to poll for new requests. Only used for "watch"
   *  mode.
   * @options host
   *  Overwrite the host to send requests to, if required. This must be the host
   *  that Drupal can use to send requests to itself.
   *
   * @throws \Exception
   */
  public function configurationExport(string $limit, array $options = [
    'pollInterval' => 15,
    'host' => '',
  ]) {
    $io = $this->io();

    $enabled = RequestHandlerController::isEnabled();
    if (!$enabled) {
      $io->error('Polling is not enabled for this site. Please go to the Advanced settings page to enable it.');
      return;
    }

    $limit_number = $limit === 'all' || $limit === 'watch' ? -1 : (int) $limit;
    $watch = $limit === 'watch';

    $request_count = 0;

    $poll_interval = (int) $options['pollInterval'];
    // Default 15.
    if (!$poll_interval) {
      $poll_interval = 15;
    }
    // Minimum 5.
    if ($poll_interval < 5) {
      $poll_interval = 5;
    }

    do {
      $request_count += RequestHandlerController::processRequests($limit_number, [
        'text' => function (string $message) use ($io) {
          $io->text($message);
        },
        'warning' => function (string $message) use ($io) {
          $io->warning($message);
        },
      ]);

      if ($watch) {
        $io->text('No more requests... polling again in ' . $poll_interval . ' seconds.');
        sleep($poll_interval);
      }
    } while ($watch);

    $io->text('Done processing ' . $request_count . ' requests.');
  }

}
