<?php

namespace Drupal\cms_content_sync\Entity;

use Drupal\cms_content_sync\Plugin\Type\EntityHandlerPluginManager;
use Drupal\cms_content_sync\PushIntent;
use Drupal\cms_content_sync\SyncCoreInterface\SyncCoreFactory;
use Drupal\Core\Config\Entity\ConfigEntityBase;

/**
 * Defines the "Content Sync - Pool" entity.
 *
 * @ConfigEntityType(
 *   id = "cms_content_sync_pool",
 *   label = @Translation("Content Sync - Pool"),
 *   handlers = {
 *     "list_builder" = "Drupal\cms_content_sync\Controller\PoolListBuilder",
 *     "form" = {
 *       "add" = "Drupal\cms_content_sync\Form\PoolForm",
 *       "edit" = "Drupal\cms_content_sync\Form\PoolForm",
 *       "delete" = "Drupal\cms_content_sync\Form\PoolDeleteForm",
 *     }
 *   },
 *   config_prefix = "pool",
 *   admin_permission = "administer cms content sync",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "backend_url",
 *   },
 *   links = {
 *     "edit-form" = "/admin/config/services/cms_content_sync/pool/{cms_content_sync_pool}/edit",
 *     "delete-form" = "/admin/config/services/cms_content_sync/synchronizations/{cms_content_sync_pool}/delete",
 *   }
 * )
 */
class Pool extends ConfigEntityBase implements PoolInterface {
  /**
   * @var string POOL_USAGE_FORBID Forbid usage of this pool for this flow
   */
  public const POOL_USAGE_FORBID = 'forbid';
  /**
   * @var string POOL_USAGE_ALLOW Allow usage of this pool for this flow
   */
  public const POOL_USAGE_ALLOW = 'allow';
  /**
   * @var string POOL_USAGE_FORCE Force usage of this pool for this flow
   */
  public const POOL_USAGE_FORCE = 'force';

  /**
   * The Pool ID.
   *
   * @var string
   */
  public $id;

  /**
   * The Pool label.
   *
   * @var string
   */
  public $label;

  /**
   * The Pool Sync Core backend URL.
   *
   * @var string
   */
  public $backend_url;

  /**
   * @var \EdgeBox\SyncCore\Interfaces\ISyncCore
   */
  protected $client;

  /**
   * @param mixed $fresh
   *
   * @return \EdgeBox\SyncCore\Interfaces\ISyncCore
   */
  public function getClient($fresh = FALSE) {
    if (!$this->client || $fresh) {
      $this->client = SyncCoreFactory::getSyncCoreV2();
    }

    return $this->client;
  }

  /**
   * Remove the http client property so that this entity can be serialized.
   */
  public function cleanToSerialize() {
    $this->client = NULL;
  }

  /**
   * Get a list of all sites from all pools that use a different version ID and
   * provide a diff on field basis.
   *
   * @param string $entity_type
   * @param string $bundle
   *
   * @throws \Exception
   *
   * @return array pool => site_id[]
   */
  public static function getAllSitesWithDifferentEntityTypeVersion($entity_type, $bundle) {
    $result = [];

    foreach (Pool::getAll() as $pool_id => $pool) {
      $diff = $pool->getClient()->getSitesWithDifferentEntityTypeVersion($pool->id, $entity_type, $bundle, Flow::getEntityTypeVersion($entity_type, $bundle));
      if (empty($diff)) {
        continue;
      }
      $result[$pool_id] = $diff;
    }

    return $result;
  }

  /**
   * Get a list of all sites for all pools that are using this entity.
   * Only works for pools that are connected to the entity on this site.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *
   * @throws \Exception
   * @throws \GuzzleHttp\Exception\GuzzleException
   *
   * @return array pool => site_id[]
   */
  public static function getAllExternalUsages($entity) {
    $entity_type = $entity->getEntityTypeId();
    $bundle = $entity->bundle();
    $entity_uuid = $entity->uuid();

    $result = [];

    foreach (EntityStatus::getInfosForEntity($entity_type, $entity_uuid) as $status) {
      $pool = $status->getPool();
      if (empty($pool)) {
        continue;
      }

      $pool_id = $pool->id;
      if (isset($result[$pool_id])) {
        continue;
      }

      $shared_entity_id = EntityHandlerPluginManager::getSharedId($entity);

      $result[$pool_id] = $pool
        ->getClient()
        ->getSyndicationService()
        ->getExternalUsages($pool_id, $entity_type, $bundle, $shared_entity_id);
    }

    return $result;
  }

  /**
   * Returns the Sync Core URL for this pool.
   *
   * @return string
   */
  public function getSyncCoreUrl() {
    return SyncCoreFactory::getSyncCoreV2Url();
  }

  /**
   * Get the newest pull/push timestamp for this pool from all status
   * entities that exist for the given entity.
   *
   * @param $entity_type
   * @param $entity_uuid
   * @param bool $pull
   * @param mixed $include_pools
   *
   * @return null|int
   */
  public function getNewestTimestamp($entity_type, $entity_uuid, $pull = TRUE, $include_pools = []) {
    $entity_status = $this->getAllStatusEntitiesFor($entity_type, $entity_uuid, $include_pools);

    $timestamp = NULL;
    foreach ($entity_status as $info) {
      $item_timestamp = $pull ? $info->getLastPull() : $info->getLastPush();
      if ($item_timestamp) {
        if (!$timestamp || $timestamp < $item_timestamp) {
          $timestamp = $item_timestamp;
        }
      }
    }

    return $timestamp;
  }

  /**
   * Get the newest pull/push timestamp for this pool from all status
   * entities that exist for the given entity.
   *
   * @param $entity_type
   * @param $entity_uuid
   * @param int $timestamp
   * @param bool $pull
   * @param mixed $include_pools
   */
  public function setTimestamp($entity_type, $entity_uuid, $timestamp, $pull = TRUE, $include_pools = []) {
    $entity_status = $this->getAllStatusEntitiesFor($entity_type, $entity_uuid, $include_pools);

    foreach ($entity_status as $info) {
      if ($pull) {
        $info->setLastPull($timestamp);
      }
      else {
        $info->setLastPush($timestamp);
      }
      $info->save();
    }
  }

  /**
   * Mark the entity as deleted in this pool (reflected on all entity status
   * entities related to this pool).
   *
   * @param $entity_type
   * @param $entity_uuid
   * @param mixed $include_pools
   */
  public function markDeleted($entity_type, $entity_uuid, $include_pools = []) {
    $entity_status = $this->getAllStatusEntitiesFor($entity_type, $entity_uuid, $include_pools);

    foreach ($entity_status as $info) {
      $info->isDeleted(TRUE);
      $info->save();
    }
  }

  /**
   * Check whether this entity has been deleted intentionally already. In this
   * case we ignore push and pull intents for it.
   *
   * @param $entity_type
   * @param $entity_uuid
   *
   * @return bool
   */
  public function isEntityDeleted($entity_type, $entity_uuid) {
    $entity_status = EntityStatus::getInfoForPool($entity_type, $entity_uuid, $this);
    foreach ($entity_status as $info) {
      if ($info->isDeleted()) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Load all cms_content_sync_pool entities.
   *
   * @return Pool[]
   */
  public static function getAll() {
    /**
     * @var Pool[] $configurations
     */
    $result = \Drupal::entityTypeManager()
      ->getStorage('cms_content_sync_pool')
      ->loadMultiple();

    ksort($result);

    return $result;
  }

  /**
   * Returns an list of pools that can be selected for an entity type.
   *
   * @oaram string $entity_type
   *  The entity type the pools should be returned for.
   *
   * @param string $bundle
   *   The bundle the pools should be returned for.
   * @param \Drupal\Core\Entity\EntityInterface $parent_entity
   *   The
   *                                                           parent entity, if any. Only required if $field_name is given-.
   * @param string $field_name
   *   The name of the parent entity field that
   *                              references this entity. In this case if the field handler is set to
   *                              "automatically push referenced entities", the user doesn't have to
   *                              make a choice as it is set automatically anyway.
   * @param mixed $entity_type
   *
   * @return array $selectable_pools
   */
  public static function getSelectablePools($entity_type, $bundle, $parent_entity = NULL, $field_name = NULL) {
    // Get all available flows.
    $flows = Flow::getAll();
    $configs = [];
    $selectable_pools = [];
    $selectable_flows = [];

    // When editing the entity directly, the "push as reference" flows won't be available and vice versa.
    $root_entity = !$parent_entity && !$field_name;
    if ($root_entity) {
      $allowed_push_options = [PushIntent::PUSH_FORCED, PushIntent::PUSH_MANUALLY, PushIntent::PUSH_AUTOMATICALLY];
    }
    else {
      $allowed_push_options = [PushIntent::PUSH_FORCED, PushIntent::PUSH_AS_DEPENDENCY];
    }

    foreach ($flows as $flow_id => $flow) {
      $flow_entity_config = $flow->getController()->getEntityTypeConfig($entity_type, $bundle);
      if (empty($flow_entity_config)) {
        continue;
      }
      if ('ignore' == $flow_entity_config['handler']) {
        continue;
      }
      if (!in_array($flow_entity_config['export'], $allowed_push_options)) {
        continue;
      }
      if ($parent_entity && $field_name) {
        $parent_flow_config = $flow->getController()->getPropertyConfig($parent_entity->getEntityTypeId(), $parent_entity->bundle(), $field_name);
        if (!$parent_flow_config) {
          continue;
        }
        if (!empty($parent_flow_config['handler_settings']['export_referenced_entities'])) {
          continue;
        }
      }

      $selectable_flows[$flow_id] = $flow;

      $configs[$flow_id] = [
        'flow_label' => $flow->label(),
        'flow' => $flow->getController()->getEntityTypeConfig($entity_type, $bundle),
      ];
    }

    foreach ($configs as $config_id => $config) {
      if (in_array('allow', $config['flow']['export_pools'])) {
        $selectable_pools[$config_id]['flow_label'] = $config['flow_label'];
        $selectable_pools[$config_id]['widget_type'] = $config['flow']['pool_export_widget_type'];
        foreach ($config['flow']['export_pools'] as $pool_id => $push_to_pool) {
          // Filter out all pools with configuration "allow".
          if (self::POOL_USAGE_ALLOW == $push_to_pool) {
            $pool_entity = \Drupal::entityTypeManager()->getStorage('cms_content_sync_pool')
              ->loadByProperties(['id' => $pool_id]);
            $pool_entity = reset($pool_entity);
            $selectable_pools[$config_id]['pools'][$pool_id] = $pool_entity->label();
          }
        }
      }
    }

    return $selectable_pools;
  }

  /**
   * Reset the status entities for this pool.
   *
   * @param string $pool_id
   *   The pool the status entities should be reset for.
   */
  public static function resetStatusEntities($pool_id = '') {
    // Reset the entity status.
    $status_storage = \Drupal::entityTypeManager()
      ->getStorage('cms_content_sync_entity_status');

    $connection = \Drupal::database();

    // For a single pool.
    if (!empty($pool_id)) {
      // Save flags to status entities that they have been reset.
      $connection->query('UPDATE cms_content_sync_entity_status SET flags=flags|:flag WHERE last_export IS NOT NULL AND pool=:pool', [
        ':flag' => EntityStatus::FLAG_LAST_PUSH_RESET,
        ':pool' => $pool_id,
      ]);
      $connection->query('UPDATE cms_content_sync_entity_status SET flags=flags|:flag WHERE last_import IS NOT NULL AND pool=:pool', [
        ':flag' => EntityStatus::FLAG_LAST_PULL_RESET,
        ':pool' => $pool_id,
      ]);

      // Actual reset.
      $db_query = $connection->update($status_storage->getBaseTable());
      $db_query->fields([
        'last_export' => NULL,
        'last_import' => NULL,
      ]);
      $db_query->condition('pool', $pool_id);
      $db_query->execute();
    }
    // For all pools.
    else {
      // Save flags to status entities that they have been reset.
      $connection->query('UPDATE cms_content_sync_entity_status SET flags=flags|:flag WHERE last_export IS NOT NULL', [
        ':flag' => EntityStatus::FLAG_LAST_PUSH_RESET,
      ]);
      $connection->query('UPDATE cms_content_sync_entity_status SET flags=flags|:flag WHERE last_import IS NOT NULL', [
        ':flag' => EntityStatus::FLAG_LAST_PULL_RESET,
      ]);

      // Actual reset.
      $db_query = $connection->update($status_storage->getBaseTable());
      $db_query->fields([
        'last_export' => NULL,
        'last_import' => NULL,
      ]);
      $db_query->execute();
    }

    // Invalidate cache by storage.
    $status_storage->resetCache();

    // Above cache clearing doesn't work reliably. So we reset the whole entity cache.
    \Drupal::service('cache.entity')->deleteAll();
  }

  /**
   * Create a pool configuration programmatically.
   *
   * @param $pool_name
   * @param string $pool_id
   * @param $backend_url
   * @param $authentication_type
   */
  public static function createPool($pool_name, $pool_id, $backend_url, $authentication_type) {
    // If no pool_id is given, create one.
    if (empty($pool_id)) {
      $pool_id = strtolower($pool_name);
      $pool_id = preg_replace('@[^a-z0-9_]+@', '_', $pool_id);
    }

    $pools = Pool::getAll();
    if (array_key_exists($pool_id, $pools)) {
      \Drupal::messenger()->addMessage('A pool with the machine name ' . $pool_id . ' does already exist. Therefor the creation has been skipped.', 'warning');
    }
    else {
      $uuid_service = \Drupal::service('uuid');
      $language_manager = \Drupal::service('language_manager');
      $default_language = $language_manager->getDefaultLanguage();

      $pool_config = \Drupal::service('config.factory')->getEditable('cms_content_sync.pool.' . $pool_id);
      $pool_config
        ->set('uuid', $uuid_service->generate())
        ->set('langcode', $default_language->getId())
        ->set('status', TRUE)
        ->set('id', $pool_id)
        ->set('label', $pool_name)
        ->set('backend_url', $backend_url)
        ->set('authentication_type', $authentication_type)
        ->save();
    }

    return $pool_id;
  }

  /**
   *
   */
  protected function getAllStatusEntitiesFor($entity_type, $entity_uuid, $include_pools) {
    if (count($include_pools) > 1) {
      $entity_status = [];
      foreach ($include_pools as $pool) {
        $entity_status = array_merge($entity_status, EntityStatus::getInfoForPool($entity_type, $entity_uuid, $pool));
      }
    }
    else {
      $entity_status = EntityStatus::getInfoForPool($entity_type, $entity_uuid, $this);
    }

    return $entity_status;
  }

}
