<?php

namespace Drupal\cms_content_sync_private_environment\Controller;

use Drupal\cms_content_sync\Controller\AuthenticationByUser;
use Drupal\cms_content_sync\SyncCoreInterface\SyncCoreFactory;
use EdgeBox\SyncCore\Interfaces\ISyncCore;
use GuzzleHttp\RequestOptions;

/**
 * Class RequestHandlerController.
 *
 * Process requests from the Sync Core asynchronously through polling.
 */
class RequestHandlerController {

  /**
   * Check whether or not we should poll for requests.
   *
   * @return bool
   */
  public static function isEnabled($set = NULL) {
    static $enabled = NULL;
    if ($set !== NULL) {
      $enabled = $set;
    }
    elseif ($enabled === NULL) {
      $enabled = SyncCoreFactory::getSyncCoreV2()->featureEnabled(ISyncCore::FEATURE_REQUEST_POLLING);
    }
    return $enabled;
  }

  /**
   * Enable request polling.
   *
   * @return void
   */
  public static function enable() {
    SyncCoreFactory::getSyncCoreV2()->enableFeature(ISyncCore::FEATURE_REQUEST_POLLING, 1);
    self::isEnabled(TRUE);
  }

  /**
   * Disable request polling.
   *
   * @return void
   */
  public static function disable() {
    SyncCoreFactory::getSyncCoreV2()->enableFeature(ISyncCore::FEATURE_REQUEST_POLLING, 0);
    self::isEnabled(FALSE);
  }

  /**
   * Process pending requests.
   *
   * @param int $limit
   *   Max number of requests to process.
   * @param array $options
   *   Optionally provide loggers.
   *
   * @return int
   */
  public static function processRequests(
    $limit = 0,
    $options = [
      'text' => NULL,
      'warning' => NULL,
    ],
  ) {
    $request = NULL;
    $request_count = 0;

    $client = SyncCoreFactory::getSyncCoreV2();

    $http_client = \Drupal::httpClient();

    $authentication_provider = AuthenticationByUser::getInstance();

    $text = function (string $message) use ($options) {
      if (!empty($options['text'])) {
        $options['text']($message);
      }
    };
    $warning = function (string $message) use ($options) {
      if (!empty($options['warning'])) {
        $options['warning']($message);
      }
    };

    do {
      $requests = $client->pollRequests();
      if (!count($requests)) {
        break;
      }
      $request_count++;

      $request = $requests[0];

      $id = $request->getId();
      $url = $request->getUrl();
      $content = $request->getInit();
      $method = $content->getMethod();
      $request_body = $content->getBody();
      $request_headers = (array) $content->getHeaders();

      // .' '.json_encode($request_headers).' '.json_encode($request_body)
      $text($request_count . '. Handling request ' . $id . ': ' . $method . ' ' . $url);

      if (!empty($options['host'])) {
        $url_parts = parse_url($url);
        $url_parts['host'] = $options['host'];

        $scheme = isset($url_parts['scheme']) ? $url_parts['scheme'] . '://' : '';
        $host = $url_parts['host'] ?? '';
        $path = $url_parts['path'] ?? '';
        $query = isset($url_parts['query']) ? '?' . $url_parts['query'] : '';

        $url = "$scheme$host$path$query";
      }

      $request_options = [
        RequestOptions::BODY => $request_body,
        RequestOptions::HEADERS => $request_headers,
        RequestOptions::AUTH => [
          $authentication_provider->getUsername(),
          $authentication_provider->getPassword(),
        ],
        RequestOptions::HTTP_ERRORS => FALSE,
      ];
      $request = $http_client->request($method, $url, $request_options);

      $response_status_code = $request->getStatusCode();
      $response_status_text = $request->getReasonPhrase();
      $response_headers = $request->getHeaders();
      $response_body = (string) $request->getBody();

      if ($response_status_code >= 200 && $response_status_code <= 299) {
        $text('Received ' . $response_status_code . ' ' . $response_status_text . '. Providing that to the Sync Core now...');
      }
      else {
        $warning('Received ' . $response_status_code . ' ' . $response_status_text . '. Providing that to the Sync Core now...');
      }

      $client->respondToRequest($id, $response_status_code, $response_status_text, $response_headers, $response_body);

      sleep(1);
    } while (!!$request && ($limit <= 0 || $request_count < $limit));

    return $request_count;
  }

  /**
   *
   */
  public static function view() {
    $waiting = SyncCoreFactory::getSyncCoreV2()->countRequestsWaitingToBePolled();
    if (!$waiting) {
      return [
        '#markup' => '<div class="messages messages--status">There are no pending requests.</div>',
      ];
    }
    return [
      '#markup' => '<div class="messages messages--warning">There are ' . $waiting . ' pending requests. Run the cron or use drush to process them.</div>',
    ];
  }

}
