<?php

namespace Drupal\cms_content_sync_draggableviews\EventSubscriber;

use Drupal\cms_content_sync\Event\AfterEntityPull;
use Drupal\cms_content_sync\Event\BeforeEntityPush;
use Drupal\cms_content_sync\Event\BeforeEntityTypeExport;
use Drupal\Core\Database\Database;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\Core\Entity\EntityTypeManager;

/**
 * Event subscriptions for events dispatched by Content Sync.
 */
class DraggableViewsSyncExtend implements EventSubscriberInterface {
  public const PROPERTY_NAME = 'draggableviews';

  /**
   * The Entity Type Manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManager
   */
  protected $entityTypeManager;

  /**
   * Just used to handle dependency injection.
   *
   * @param \Drupal\Core\Entity\EntityTypeManager $entityTypeManager
   *   The Drupal core entity type manager.
   */
  public function __construct(EntityTypeManager $entityTypeManager) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * Returns an array of event names this subscriber wants to listen to.
   *
   * @return array
   *   The event names to listen to
   */
  public static function getSubscribedEvents() {
    $events[BeforeEntityPush::EVENT_NAME][] = ['extendPush'];
    $events[AfterEntityPull::EVENT_NAME][] = ['extendPull'];
    $events[BeforeEntityTypeExport::EVENT_NAME][] = ['extendEntityType'];

    return $events;
  }

  /**
   * Add the field to the entity type for the draggable views weight config.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function extendEntityType(BeforeEntityTypeExport $event) {
    if (!$this->supportsEntityType($event->getEntityTypeName())) {
      return;
    }

    $value_definition = $event
      ->getDefinition()
      ->addObjectProperty(self::PROPERTY_NAME, 'Draggable views', FALSE, FALSE, 'draggableviews')
      ->addObjectProperty('*', 'View name', FALSE, FALSE, 'draggableviews_view_settings')
      ->addObjectProperty('*', 'Display name', FALSE, FALSE, 'draggableviews_view_display_settings');
    $value_definition->addStringProperty('args', 'Arguments', FALSE, FALSE, 'string');
    $value_definition->addIntegerProperty('weight', 'Weight', FALSE, TRUE, 'integer');
    $value_definition->addIntegerProperty('parent', 'Parent', FALSE, TRUE, 'integer');
  }

  /**
   * Alter the push to include the draggable views settings.
   *
   * If enabled for the entity type.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function extendPush(BeforeEntityPush $event) {
    $intent = $event->intent;
    $entity = $event->entity;

    if (!$this->supportsEntityType($entity->getEntityTypeId())) {
      return;
    }

    $connection = Database::getConnection();
    $query = $connection->select('draggableviews_structure', 'dvs');
    $query->condition('dvs.entity_id', $entity->id());
    $query->fields('dvs', [
      'view_name',
      'view_display',
      'args',
      'weight',
      'parent',
    ]);
    $result = $query->execute();
    $result->setFetchMode(\PDO::FETCH_ASSOC);
    $result = $result->fetchAll();

    $values = [];
    foreach ($result as $row) {
      $values[$row['view_name']][$row['view_display']] = [
        'args' => $row['args'],
        'weight' => $row['weight'],
        'parent' => 0,
      ];
    }

    if (!count($values)) {
      $intent->setProperty(self::PROPERTY_NAME, NULL);

      return;
    }

    $intent->setProperty(self::PROPERTY_NAME, $values);
  }

  /**
   * Extend the entity pull to handle draggable views related entities.
   *
   * @internal param $entity
   * @internal param $intent
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function extendPull(AfterEntityPull $event) {
    $intent = $event->getIntent();
    $entity = $event->getEntity();

    if (!$this->supportsEntityType($entity->getEntityTypeId())) {
      return;
    }

    $values = $intent->getProperty(self::PROPERTY_NAME);

    if (empty($values)) {
      return;
    }

    $connection = Database::getConnection();

    foreach ($values as $view_name => $view_values) {
      foreach ($view_values as $view_display => $display_values) {
        // Remove old data.
        $connection->delete('draggableviews_structure')
          ->condition('view_name', $view_name)
          ->condition('view_display', $view_display)
          ->condition('entity_id', $entity->id())
          ->execute();

        // Add new data.
        $record = [
          'view_name' => $view_name,
          'view_display' => $view_display,
          'args' => $display_values['args'],
          'entity_id' => $entity->id(),
          'weight' => $display_values['weight'],
          'parent' => $display_values['parent'],
        ];

        $connection
          ->insert('draggableviews_structure')
          ->fields($record)
          ->execute();
      }
    }
  }

  /**
   * Check if the entity type can be displayed in a draggable view.
   *
   * @param string $entity_type_name
   *   The name of the entity type.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *
   * @return bool
   *   Whether or not the simple sitemap module supports configuring sitemap
   *   settings for the given entity type + bundle.
   */
  protected function supportsEntityType($entity_type_name) {
    $entityTypeManager = $this->entityTypeManager;
    $entity_type = $entityTypeManager->getDefinition($entity_type_name, FALSE);

    if (!$entity_type instanceof ContentEntityTypeInterface) {
      return FALSE;
    }

    return TRUE;
  }

}
