<?php

namespace Drupal\cms_content_sync_developer\EventSubscriber;

use Drupal\cms_content_sync\Entity\Flow;
use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Link;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * A subscriber triggering a config when certain configuration changes.
 */
class VersionWarning implements EventSubscriberInterface {

  /**
   * The config Factory.
   *
   * @var \Drupal\Core\Config\ConfigFactory
   */
  protected $configFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactory $configFactory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(ConfigFactory $configFactory, AccountProxyInterface $currentUser, MessengerInterface $messenger) {
    $this->configFactory = $configFactory;
    $this->currentUser = $currentUser;
    $this->messenger = $messenger;
  }

  /**
   * Show version warning.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The Event to process.
   */
  public function showVersionWarning(RequestEvent $event) {
    $current_user = $this->currentUser;
    if ($current_user->hasPermission('administer cms content sync')) {
      // Old Flow type.
      $config = $this->configFactory;
      $messenger = $this->messenger;
      $developer_config = $config->getEditable('cms_content_sync.developer');
      $version_mismatch = $developer_config->get('version_mismatch');
      if (!empty($version_mismatch)) {
        $links = [];
        foreach ($version_mismatch as $flow_id => $flow) {
          if (Flow::VARIANT_PER_BUNDLE === Flow::getAll(FALSE)[$flow_id]->variant) {
            $links[$flow_id] = Link::fromTextAndUrl($flow_id, Url::fromRoute('entity.cms_content_sync_flow.edit_form', ['cms_content_sync_flow' => $flow_id], ['absolute' => TRUE]))->toString();
          }
        }

        if (count($links)) {
          $mismatching_flow_labels = implode(', ', $links);
          $message = new TranslatableMarkup('You have to update the related flow(s) @flows to keep the content synchronization intact. Failing to update the config may break the synchronization and lead to damaged or missing content.', ['@flows' => new FormattableMarkup($mismatching_flow_labels, [])]);
          $messenger->addWarning($message);
        }

        // New Flow type.
        $links = [];
        foreach (Flow::getAll() as $flow) {
          if (Flow::VARIANT_SIMPLE === $flow->variant && $flow->getController()->needsEntityTypeUpdate()) {
            $links[$flow_id] = Link::fromTextAndUrl($flow_id, Url::fromRoute('entity.cms_content_sync_flow.edit_form', ['cms_content_sync_flow' => $flow_id], ['absolute' => TRUE]))->toString();
          }
        }
        if (count($links)) {
          $mismatching_flow_labels = implode(', ', $links);
          $message = new TranslatableMarkup('You have to export the related flow(s) @flows to keep the content synchronization intact. Failing to export the Flows may break the synchronization and lead to damaged or missing content.', ['@flows' => new FormattableMarkup($mismatching_flow_labels, [])]);
          $messenger->addWarning($message);
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    $events[KernelEvents::REQUEST][] = ['showVersionWarning'];

    return $events;
  }

}
