<?php

namespace Drupal\cms_content_sync\Plugin\rest\resource;

use Drupal\cms_content_sync\Entity\EntityStatus;
use Drupal\cms_content_sync\Entity\Flow;
use Drupal\cms_content_sync\Exception\SyncException;
use Drupal\cms_content_sync\Plugin\Type\EntityHandlerPluginManager;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfo;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\Core\Render\Renderer;
use Drupal\rest\ModifiedResourceResponse;
use Drupal\rest\Plugin\ResourceBase;
use EdgeBox\SyncCore\Interfaces\IApplicationInterface;
use EdgeBox\SyncCore\V2\Raw\Model\RemoteEntityListRequestMode;
use EdgeBox\SyncCore\V2\Raw\Model\RemoteEntityListResponse;
use EdgeBox\SyncCore\V2\Raw\Model\RemoteEntitySummary;
use EdgeBox\SyncCore\V2\Raw\Model\RemoteEntityTranslationDetails;
use EdgeBox\SyncCore\V2\Raw\Model\RemoteRequestQueryParamsEntityList;
use PDO;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides entity interfaces for Content Sync, allowing Sync Core v2 to
 * list entities.
 *
 * @RestResource(
 *   id = "cms_content_sync_sync_core_entity_list",
 *   label = @Translation("Content Sync: Sync Core: Entity list"),
 *   uri_paths = {
 *     "canonical" = "/rest/cms-content-sync/v2/{flow_id}"
 *   }
 * )
 */
class SyncCoreEntityListResource extends ResourceBase
{
    protected const NONE = 'null';

    /**
     * @var \Drupal\Core\Entity\EntityTypeBundleInfo
     */
    protected $entityTypeBundleInfo;

    /**
     * @var \Drupal\Core\Entity\EntityTypeManager
     */
    protected $entityTypeManager;

    /**
     * @var \Drupal\Core\Render\Renderer
     */
    protected $renderedManager;

    /**
     * @var \Drupal\Core\Entity\EntityRepositoryInterface
     */
    protected $entityRepository;

    /**
     * Constructs an object.
     *
     * @param array                                          $configuration
     *                                                                                A configuration array containing information about the plugin instance
     * @param string                                         $plugin_id
     *                                                                                The plugin_id for the plugin instance
     * @param mixed                                          $plugin_definition
     *                                                                                The plugin implementation definition
     * @param array                                          $serializer_formats
     *                                                                                The available serialization formats
     * @param \Psr\Log\LoggerInterface                       $logger
     *                                                                                A logger instance
     * @param \Drupal\Core\Entity\EntityTypeBundleInfo       $entity_type_bundle_info
     *                                                                                An entity type bundle info instance
     * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
     *                                                                                An entity type manager instance
     * @param \Drupal\Core\Render\Renderer                   $render_manager
     *                                                                                A rendered instance
     * @param \Drupal\Core\Entity\EntityRepositoryInterface  $entity_repository
     *                                                                                The entity repository interface
     */
    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        array $serializer_formats,
        LoggerInterface $logger,
        EntityTypeBundleInfo $entity_type_bundle_info,
        EntityTypeManagerInterface $entity_type_manager,
        Renderer $render_manager,
        EntityRepositoryInterface $entity_repository
    ) {
        parent::__construct(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $serializer_formats,
            $logger
        );

        $this->entityTypeBundleInfo = $entity_type_bundle_info;
        $this->entityTypeManager = $entity_type_manager;
        $this->renderedManager = $render_manager;
        $this->entityRepository = $entity_repository;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(
        ContainerInterface $container,
        array $configuration,
        $plugin_id,
        $plugin_definition
    ) {
        return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('rest'),
      $container->get('entity_type.bundle.info'),
      $container->get('entity_type.manager'),
      $container->get('renderer'),
      $container->get('entity.repository')
    );
    }

    public function get($flow_id)
    {
        $query = \Drupal::request()->query->all();
        $queryObject = new RemoteRequestQueryParamsEntityList($query);

        $entity_type = $queryObject->getNamespaceMachineName();
        $bundle = $queryObject->getMachineName();

        $flow = Flow::getAll()[$flow_id] ?? null;
        if (empty($flow)) {
            if (IApplicationInterface::FLOW_NONE === $flow_id) {
                $flow = new Flow([
                    'id' => $flow_id,
                    'name' => 'Virtual',
                    'type' => Flow::TYPE_PUSH,
                    'variant' => Flow::VARIANT_SIMPLE,
                    'simple_settings' => [
                        'poolAssignment' => 'force',
                        'mode' => 'automatically',
                        'deletions' => false,
                        'updateBehavior' => 'ignore',
                        'ignoreUnpublishedChanges' => false,
                        'allowExplicitUnpublishing' => false,
                        'pushMenuItems' => false,
                        'pushPreviews' => false,
                        'mergeLocalChanges' => false,
                        'resolveUserReferences' => 'name',
                        'poolSelectionWidget' => 'checkboxes',
                        'entityTypeSettings' => [
                            $entity_type => [
                                'perBundle' => [
                                    $bundle => [
                                        'mode' => 'default',
                                    ],
                                ],
                            ],
                        ],
                    ],
                ], 'cms_content_sync_flow');
                $flow->getController()->isVirtual(true);
            } else {
                $message = t("The flow @flow_id doesn't exist.", ['@flow_id' => $flow_id])->render();
                \Drupal::logger('cms_content_sync')->notice('@not LIST: @message', [
                    '@not' => 'NO',
                    '@message' => $message,
                ]);

                return $this->respondWith(
                    ['message' => $message],
                    404
                );
            }
        }

        $page = (int) $queryObject->getPage();
        if (!$page) {
            $page = 0;
        }

        $items_per_page = (int) $queryObject->getItemsPerPage();
        if (!$items_per_page) {
            $items_per_page = 0;
        }

        $mode = $queryObject->getMode();
        if (!$mode) {
            return $this->returnError(t('The mode query parameter is required.')->render());
        }

        // Need to convert miliseconds to seconds.
        $changed_after = $queryObject->getChangedAfter() ? floor((int) $queryObject->getChangedAfter() / 1000) : null;

        $search = empty($query['search']) ? null : $query['search'];

        $skip = $page * $items_per_page;

        $database = \Drupal::database();

        /**
         * @var RemoteEntitySummary[] $items
         */
        $items = [];

        try {
            // If ALL entities are requested, we can't rely on the status entity.
            // Instead, we query for these entities by their type's table directly.
            if (RemoteEntityListRequestMode::ALL === $mode) {
                if (!$entity_type || !$bundle) {
                    return $this->returnError(t("The type and bundle query parameters are required for mode 'all'.")->render());
                }

                $entity_type_storage = \Drupal::entityTypeManager()->getStorage($entity_type);
                $bundle_key = $entity_type_storage->getEntityType()->getKey('bundle');
                $id_key = $entity_type_storage->getEntityType()->getKey('id');
                $base_table = $entity_type_storage->getBaseTable();
                $data_table = $entity_type_storage->getDataTable();
                $definitions = \Drupal::service('entity_field.manager')->getFieldDefinitions($entity_type, $bundle);

                $query = $database->select($base_table, 'bt');
                $query
                    ->condition('bt.'.$bundle_key, $bundle)
                    ->fields('bt', [$id_key]);

                // Join data table.
                $query->join($data_table, 'dt', 'dt.'.$id_key.'= bt.'.$id_key);

                $label_property = $entity_type_storage->getEntityType()->getKey('label');
                if (isset($search, $label_property)) {
                    $query
                        ->condition('dt.'.$label_property, '%'.$database->escapeLike($search).'%', 'LIKE');
                }

                // Ignore unpublished entities based on the flow configuration.
                $entity_type_config = $flow->getController()->getEntityTypeConfig($entity_type, $bundle);
                $handler_settings = $entity_type_config['handler_settings'];
                $status_key = $entity_type_storage->getEntityType()->getKey('status');
                if (true == $handler_settings['ignore_unpublished'] && isset($status_key)) {
                    if (true == $handler_settings['allow_explicit_unpublishing']) {
                        // Join the entity status table to check if the entity has been exported before to allow explizit unpublishing.
                        $query->join('cms_content_sync_entity_status', 'cses', 'cses.entity_uuid = bt.uuid');

                        // If status is 0 and the entity has been exported before.
                        $and = $query->andConditionGroup()
                            ->condition('dt.'.$status_key, '0')
                            ->condition('cses.last_export', 0, '>');

                        $or = $query->orConditionGroup()
                            ->condition($and)
                            ->condition('dt.'.$status_key, '1');

                        $query->condition($or);
                    } else {
                        $query
                            ->condition('dt.'.$status_key, '1');
                    }
                }

                if (isset($definitions['created'])) {
                    if ($changed_after) {
                        $query
                            ->condition('dt.created', $changed_after, '>');
                    }
                    $query
                        ->orderBy('dt.created', 'ASC');
                } else {
                    $query->orderBy('bt.'.$id_key, 'ASC');
                }

                $total_number_of_items = (int) $query->countQuery()->execute()->fetchField();

                if ($total_number_of_items && $items_per_page) {
                    $ids = $query
                        ->range($skip, $items_per_page)
                        ->execute()
                        ->fetchAll(PDO::FETCH_COLUMN);
                    $entities = $entity_type_storage->loadMultiple($ids);
                    foreach ($entities as $entity) {
                        $items[] = $this->getItem($flow, $entity, EntityStatus::getInfosForEntity($entity->getEntityTypeId(), $entity->uuid(), ['flow' => $flow_id]));
                    }
                }
            } else {
                $query = $database->select('cms_content_sync_entity_status', 'cses');

                if ($entity_type && $bundle) {
                    $entity_type_storage = \Drupal::entityTypeManager()->getStorage($entity_type);
                    $bundle_key = $entity_type_storage->getEntityType()->getKey('bundle');
                    $table = $entity_type_storage->getBaseTable();
                    $query->join($table, 'bt', 'bt.uuid = cses.entity_uuid');
                }

                $query
                    ->condition('cses.flow', $flow_id);

                $changed_field = RemoteEntityListRequestMode::PULLED === $mode ? 'last_import' : 'last_export';
                if ($changed_after) {
                    $query->condition('cses.'.$changed_field, $changed_after, '>');
                } elseif (RemoteEntityListRequestMode::PULLED === $mode || RemoteEntityListRequestMode::PUSHED === $mode) {
                    $query->condition('cses.'.$changed_field, 0, '>');
                }

                if ($entity_type) {
                    $query
                        ->condition('cses.entity_type', $entity_type);
                    if ($bundle && !empty($bundle_key)) {
                        $query
                            ->condition('bt.'.$bundle_key, $bundle);
                    }
                }

                if (RemoteEntityListRequestMode::PUSH_FAILED === $mode) {
                    $query
                        ->where('flags&:flag=:flag', [':flag' => EntityStatus::FLAG_PUSH_FAILED]);
                }

                $query->addExpression('MIN(cses.id)', 'min_id');
                $query
                    ->orderBy('min_id', 'ASC')
                    ->fields('cses', ['entity_type', 'entity_uuid']);

                $query->groupBy('cses.entity_type');
                $query->groupBy('cses.entity_uuid');

                $total_number_of_items = (int) $query->countQuery()->execute()->fetchField();

                if ($total_number_of_items && $items_per_page) {
                    $query->range($skip, $items_per_page);
                    $ids = $query->execute()->fetchAll(PDO::FETCH_ASSOC);
                    foreach ($ids as $id) {
                        $entity = \Drupal::service('entity.repository')->loadEntityByUuid(
                            $id['entity_type'],
                            $id['entity_uuid']
                        );

                        $items[] = $this->getItem($flow, $entity, EntityStatus::getInfosForEntity($id['entity_type'], $id['entity_uuid'], ['flow' => $flow_id]));
                    }
                }
            }

            if (!$items_per_page) {
                $number_of_pages = $total_number_of_items;
            } else {
                $number_of_pages = ceil($total_number_of_items / $items_per_page);
            }

            $result = new RemoteEntityListResponse();
            $result->setPage($page);
            $result->setNumberOfPages($number_of_pages);
            $result->setItemsPerPage($items_per_page);
            $result->setTotalNumberOfItems($total_number_of_items);
            $result->setItems($items);

            $body = $result->jsonSerialize();

            // Turn object into array because Drupal doesn't think stdObject can be
            // serialized the same way.
            return $this->respondWith(json_decode(json_encode($body), true), SyncCoreEntityItemResource::CODE_OK);
        } catch (SyncException $e) {
            $message = $e->getSyncExceptionMessage();
            \Drupal::logger('cms_content_sync')->notice('@not LIST: @message', [
                '@not' => 'NO',
                '@message' => $message,
            ]);

            return $this->respondWith(
                $e->serialize(),
                SyncCoreEntityItemResource::CODE_INTERNAL_SERVER_ERROR
            );
        } catch (\Exception $e) {
            \Drupal::logger('cms_content_sync')->notice('@not LIST: @message', [
                '@not' => 'NO',
                '@message' => $e->getMessage(),
            ]);

            return $this->respondWith(
                [
                    'message' => 'Unexpected error: '.$e->getMessage(),
                    'stack' => $e->getTraceAsString(),
                ],
                SyncCoreEntityItemResource::CODE_INTERNAL_SERVER_ERROR
            );
        }
    }

    protected function respondWith($body, $status)
    {
        return new ModifiedResourceResponse(
            $body,
            $status
        );
    }

    protected function returnError($message, $code = SyncCoreEntityItemResource::CODE_BAD_REQUEST)
    {
        \Drupal::logger('cms_content_sync')->notice('@not LIST: @message', [
            '@not' => 'NO',
            '@message' => $message,
        ]);

        return $this->respondWith(
            ['message' => $message],
            $code
        );
    }

    protected function getItem(?Flow $flow, ?object $entity, array $statuses)
    {
        $item = new RemoteEntitySummary();

        $pools = [];
        $last_push = null;
        foreach ($statuses as $status) {
            $pools[] = $status->get('pool')->value;
            if ($status->getLastPush()) {
                if (!$last_push || $status->getLastPush() > $last_push->getLastPush()) {
                    $last_push = $status;
                }
            }
        }

        $item->setPoolMachineNames($pools);

        $item->setIsSource(empty($statuses) || (bool) $last_push);

        if ($entity) {
            $item->setEntityTypeNamespaceMachineName($entity->getEntityTypeId());
            $item->setEntityTypeMachineName($entity->bundle());
            $item->setEntityTypeVersion(Flow::getEntityTypeVersion($entity->getEntityTypeId(), $entity->bundle()));
            $item->setRemoteUuid($entity->uuid());
            if (EntityHandlerPluginManager::mapById($entity->getEntityType())) {
                $item->setRemoteUniqueId($entity->id());
            }
            $item->setLanguage($entity->language()->getId());
            $item->setName($entity->label());

            $item->setIsDeleted(false);

            if ($flow) {
                $config = $flow->getController()->getEntityTypeConfig($entity->getEntityTypeId(), $entity->bundle());
                $handler = $flow->getController()->getEntityTypeHandler($entity->getEntityTypeId(), $entity->bundle(), $config);
            } else {
                $entity_plugin_manager = \Drupal::service('plugin.manager.cms_content_sync_entity_handler');
                $entity_handlers = $entity_plugin_manager->getHandlerOptions($entity->getEntityTypeId(), $entity->bundle(), true);
                $entity_handler_names = array_keys($entity_handlers);
                $handler_id = reset($entity_handler_names);
                $handler = $entity_plugin_manager->createInstance($handler_id, [
                    'entity_type_name' => $entity->getEntityTypeId(),
                    'bundle_name' => $entity->bundle(),
                    'settings' => [],
                    'sync' => null,
                ]);
            }
            $item->setViewUrl($handler->getViewUrl($entity));

            if ($flow && $entity instanceof TranslatableInterface) {
                $translations = [];
                $config = $flow->getController()->getEntityTypeConfig($entity->getEntityTypeId(), $entity->bundle());
                $handler = $flow->getController()->getEntityTypeHandler($entity->getEntityTypeId(), $entity->bundle(), $config);
                foreach ($entity->getTranslationLanguages(false) as $language) {
                    $translation_dto = new RemoteEntityTranslationDetails();
                    $translation_dto->setLanguage($language->getId());
                    $view_url = $handler->getViewUrl($entity->getTranslation($language->getId()));
                    $translation_dto->setViewUrl($view_url);
                    $translations[] = $translation_dto;
                }
                $item->setTranslations($translations);
            }
        } else {
            $status = $last_push ? $last_push : reset($statuses);
            $item->setEntityTypeNamespaceMachineName($status->getEntityTypeName());
            $item->setEntityTypeVersion($status->getEntityTypeVersion());
            $item->setRemoteUuid($status->getUuid());

            $item->setIsDeleted($status->isDeleted());
        }

        return $item;
    }
}
